// SPDX-License-Identifier: MIT
pragma solidity ^0.8.20;

import "@openzeppelin/contracts/token/ERC20/IERC20.sol";
import "@openzeppelin/contracts/access/IAccessControl.sol";

/**
 * @title IKHYPE
 * @notice Interface for the kHYPE token contract
 * @dev This is a simplified interface for integration purposes.
 *      For the full interface, see src/interfaces/IKHYPE.sol
 */
interface IKHYPE is IERC20, IAccessControl {
    /**
     * @notice Mints new tokens to the specified address
     * @dev Only callable by addresses with MINTER_ROLE
     * @param to Address to receive the minted tokens
     * @param amount Amount of tokens to mint
     */
    function mint(address to, uint256 amount) external;

    /**
     * @notice Burns tokens from the specified address
     * @dev Only callable by addresses with BURNER_ROLE
     * @param from Address to burn tokens from
     * @param amount Amount of tokens to burn
     */
    function burn(address from, uint256 amount) external;

    /**
     * @notice EIP-2612 permit for gasless approvals
     * @param owner Token owner
     * @param spender Address to approve
     * @param value Amount to approve
     * @param deadline Signature expiry timestamp
     * @param v Signature component
     * @param r Signature component
     * @param s Signature component
     */
    function permit(
        address owner,
        address spender,
        uint256 value,
        uint256 deadline,
        uint8 v,
        bytes32 r,
        bytes32 s
    ) external;

    /// @notice Role identifier for addresses allowed to mint tokens
    function MINTER_ROLE() external view returns (bytes32);

    /// @notice Role identifier for addresses allowed to burn tokens
    function BURNER_ROLE() external view returns (bytes32);
}

