// SPDX-License-Identifier: MIT
pragma solidity ^0.8.20;

/**
 * @title IStakingManager
 * @notice Interface for the Kinetiq StakingManager contract
 * @dev This is a simplified interface for integration purposes.
 *      For the full interface, see src/interfaces/IStakingManager.sol
 */
interface IStakingManager {
    /* ========== STRUCTS ========== */

    struct WithdrawalRequest {
        uint256 hypeAmount;   // Amount in HYPE to withdraw
        uint256 kHYPEAmount;  // Amount in kHYPE to burn (excluding fee)
        uint256 kHYPEFee;     // Fee amount in kHYPE tokens
        uint256 bufferUsed;   // Amount fulfilled from hypeBuffer
        uint256 timestamp;    // Request timestamp
    }

    /* ========== EVENTS ========== */

    event StakeReceived(address indexed staking, address indexed staker, uint256 amount);
    event WithdrawalQueued(
        address indexed staking,
        address indexed user,
        uint256 indexed withdrawalId,
        uint256 kHYPEAmount,
        uint256 hypeAmount,
        uint256 feeAmount
    );
    event WithdrawalConfirmed(address indexed user, uint256 indexed withdrawalId, uint256 amount);
    event InstantUnstakeExecuted(
        address indexed user,
        uint256 kHYPEAmount,
        uint256 hypeReceived,
        uint256 kHYPEFee,
        uint256 feeRateBps,
        uint256 kHYPEFeeBurned,
        uint256 kHYPEFeeToTreasury
    );

    /* ========== VIEW FUNCTIONS ========== */

    /// @notice Gets the total amount of HYPE staked
    function totalStaked() external view returns (uint256);

    /// @notice Gets the current HYPE buffer amount
    function hypeBuffer() external view returns (uint256);

    /// @notice Gets the minimum stake amount per transaction
    function minStakeAmount() external view returns (uint256);

    /// @notice Gets the minimum withdrawal amount
    function minWithdrawalAmount() external view returns (uint256);

    /// @notice Gets the withdrawal delay period
    function withdrawalDelay() external view returns (uint256);

    /// @notice Gets the quick withdrawal delay period (for buffer-only withdrawals)
    function quickWithdrawalDelay() external view returns (uint256);

    /// @notice Gets withdrawal request details for a user
    function withdrawalRequests(address user, uint256 id) external view returns (WithdrawalRequest memory);

    /// @notice Gets the next withdrawal ID for a user
    function nextWithdrawalId(address user) external view returns (uint256);

    /// @notice Gets the current unstake fee rate in basis points
    function unstakeFeeRate() external view returns (uint256);

    /// @notice Gets the basis points constant (10000 = 100%)
    function BASIS_POINTS() external view returns (uint256);

    /// @notice Gets the instant unstake pool address
    function instantUnstakePool() external view returns (address);

    /* ========== MUTATIVE FUNCTIONS ========== */

    /// @notice Stakes HYPE tokens and receives kHYPE in return
    function stake() external payable;

    /// @notice Queues a withdrawal request
    /// @param kHYPEAmount Amount of kHYPE to withdraw
    function queueWithdrawal(uint256 kHYPEAmount) external;

    /// @notice Confirms a withdrawal request after the delay period
    /// @param withdrawalId The ID of the withdrawal to confirm
    function confirmWithdrawal(uint256 withdrawalId) external;

    /**
     * @notice Instant unstake - withdraw HYPE immediately from buffer with flat fee
     * @param kHYPEAmount Amount of kHYPE to unstake
     * @param minHYPEOut Minimum HYPE to receive (slippage protection)
     */
    function instantUnstake(uint256 kHYPEAmount, uint256 minHYPEOut) external;
}

